/** @file   carnivorealien.cpp
 * @brief   Implementation of CarnivoreAlien - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "carnivorealien.h" // class's header file
#include "carnivoreai.h"
#include "animplayer.h"
#include "GfxManager.h"
#include "warglobals.h"
#include "soundsamples.h"
#include "settings.h"
#include "player.h"
#include "AnimId.h"
using namespace::eng2d;

namespace WeWantWar {

/** Constructor
 */
CarnivoreAlien::CarnivoreAlien() :
  Alien( ),
  m_spdVec( 0, 0 ),
  m_damagePoints( 0 )
{
  ObjectID::Type oid = ObjectID::TYPE_CARNIVOREALIEN;
  
  m_spdVec.y( - Settings::floatObjProp(oid, "spd_walk:"));
                                                
	this->setCorrectAnimation( GameAnims::EIdle );

  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setProperties( GameObject::PROP_GRENADEFORCE );
  
  // 4 collisionpoints
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  m_damagePoints = Settings::intObjProp(oid, "damagepoints:");
  
  // Create the ai-object.
  this->setController( new CarnivoreAI( this ) );
}



/** Destructor.
 */
CarnivoreAlien::~CarnivoreAlien()
{
}



/** The update-method.
 */
void CarnivoreAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    // We're dying, so update the dying animation
    if ( AnimatedObject::m_animation.paused() ) {
      // The dying animation has been run through, so set this object
      // to killed-state
      this->state( STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
      return;
    }

  } else {
    // Update the controller and do what it tells us to do.
    m_pController->update();
    
    if ( m_pController->shoot() && this->reloading() == false ) {
      // Attack the player.
      this->attack();
      this->setCorrectAnimation( GameAnims::EShoot );
    }


    if ( m_pController->shoot() == 0 && this->getAnimID() == GameAnims::EShoot ) {
      // The alien is no longer attackin player. Set the default animation
      // back
      this->setCorrectAnimation( GameAnims::EWalk );
    }

    // If we should walk, we walk
    Vec2D v = m_spdVec;
    v.rotate( this->angle() );
    if ( m_pController->forward() != 0 ) {
      if ( m_pController->forward() == 1 ) {
        this->move( v );
        this->setCorrectAnimation( GameAnims::EWalk );
        
      } else if ( m_pController->forward() == 2 ) {
        v.scale( 1.4 );
        this->move( v );
        
      }
      
    }
    
    this->changeAngle( m_pController->turn() );
  }
}



/** Kills this alien.
 */
void CarnivoreAlien::kill()
{

  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    // This alien is already dying.
    return;
  }
  this->state( GameObject::STATE_DYING );

  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Makes sound.
 */
void CarnivoreAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_PAIN ) {
    int t = ( rand() % 2 < 1 ) ? SMP_ALIEN1_PAIN : SMP_ALIEN1_PAIN2;
    Sound::playSample( t, false );
    
  } else if ( id == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN1_DIE, false );
  }
}



/** Hit by bullet
 */
bool CarnivoreAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      int partCount = 12;
      if ( pB->iType == Bullet::ESniperRifle ) {
        partCount = 20;
      }
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                             partCount, Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
      
      AIController* pC = dynamic_cast< AIController* >( this->getController() );
      AIController::BFlag flags = pC->getFlags();
      flags &= ~AIController::WAIT_TARGET;
      pC->setFlags( flags );
    }
  }
  return ret;
}



/** Returns the type of this object.
 */
ObjectID::Type CarnivoreAlien::objectType() const
{
  return ObjectID::TYPE_CARNIVOREALIEN;
}



/** Sets the correct animation.
 */
void CarnivoreAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KCarnivoreAlien,
                                                      aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Attacks
 */
void CarnivoreAlien::attack()
{
  AIController* pAIC = dynamic_cast<AIController*>( m_pController );
  GameObject* pTO = pAIC->getTarget();

  Vec2D plaPos = pTO->position();
  Vec2D dirV = plaPos - this->position();
  ParticleBlood* pPB = new ParticleBlood( plaPos, dirV, 10, Color(140,70,30) );
  WarGlobals::pPartManager->addSystem( pPB );

  bool died = pTO->causeDamage( m_damagePoints );
  if ( died ) {
    // Kill the player
    pTO->kill();
  }

  // We wait a moment until we're able to reattack
  this->setCounter( RELOAD_COUNTER_INDEX, 9 );
}

} // end of namespace
